//////////////////////////////////////////////////////////
//
// Filename: tms.cpp
// Author:   Stefan (Shaderman) Greven
// Date:     29.12.2009
//
// Description: This is the main Torque Master Server code
//
//////////////////////////////////////////////////////////

#include "tms.h"
#include "logging.h"
#include "packet.h"
#include "options.h"
#include "sip_store.h"

#include <iostream>

#include <boost/algorithm/string.hpp>
#include <boost/thread.hpp>

extern struct prog_opt ProgramOptions;

CMaster::CMaster(CSipStore& sip_store, boost::asio::io_service& ios_socket, boost::asio::io_service& ios_cleanupTimer,
                 boost::asio::io_service& ios_execTimer, boost::asio::ip::address ip_addr, short port)
    : m_sipStore(sip_store), m_socketIos(ios_socket), m_socket(ios_socket, udp::endpoint(ip_addr, port)),
      m_cleanupTimer(ios_cleanupTimer), m_execTimer(ios_execTimer)
{
    // initialize the cleanup timer
    m_cleanupTimer.expires_from_now(boost::posix_time::seconds(ProgramOptions.interval));
    m_cleanupTimer.async_wait(boost::bind(&CMaster::runCleanup, this));

    // initialize the exec timer
    m_execTimer.expires_from_now(boost::posix_time::minutes(ProgramOptions.execinterval));
    m_execTimer.async_wait(boost::bind(&CMaster::runExec, this));

    // get ready to receive data
    m_socket.async_receive_from(
        boost::asio::buffer(m_packet), m_sender_endpoint, boost::bind(&CMaster::checkPacket,
        this, boost::asio::placeholders::error, boost::asio::placeholders::bytes_transferred)
    );

    BOOST_LOG_SEV(slg, I) << "[I] Listening on " << ip_addr.to_string() << ":" << port << ".";

#if defined(MASTER_DEBUG)
    addFakeServers(3);
#endif
}

void CMaster::packetSent(const boost::system::error_code& /*error*/, size_t /*bytes_sent*/)
{
    // This unused function is called when a package was sent
}

void CMaster::checkPacket(const boost::system::error_code& error, size_t bytes_recvd)
{
    if (!error && bytes_recvd > 0)
    {
        m_packet.prepare();

        std::string senderIp    = m_sender_endpoint.address().to_string();
        U32 senderPort          = m_sender_endpoint.port();

        switch(m_packet.readU8())
        {
            case MasterServerGameTypesRequest:
                BOOST_LOG_SEV(slg, D) << "[R] Game types request from " << m_sender_endpoint;
                handleGameTypesRequest();
                break;

            case MasterServerListRequest:
                BOOST_LOG_SEV(slg, D) << "[R] Server list request from " << m_sender_endpoint;
                handleServerListRequest();
                break;

            case GameHeartbeat:
                BOOST_LOG_SEV(slg, D) << "[R] Heartbeat from " << m_sender_endpoint;
                handleHeartbeat(senderIp, senderPort);
                sendInfoRequest(senderIp, senderPort);
                break;

            case GameMasterInfoResponse:
                BOOST_LOG_SEV(slg, D) << "[R] Info response from " << m_sender_endpoint;
                handleInfoResponse(senderIp, senderPort);
                break;

            default:
                BOOST_LOG_SEV(slg, D) << "[R] Unknown packet from " << m_sender_endpoint;
        }
    }

    m_socket.async_receive_from(
        boost::asio::buffer(m_packet), m_sender_endpoint, boost::bind(&CMaster::checkPacket,
        this, boost::asio::placeholders::error, boost::asio::placeholders::bytes_transferred)
    );
}

void CMaster::handleHeartbeat(std::string ip, U32 port)
{
    bool    ret;
    U32     key;

    ret = false;

    key = getRandU32();

    ret = m_sipStore.isListed(ip, port);

    if(ret)
    {
        // update the timestamp
        m_sipStore.setTimeStampAndKey(ip, port, key);
        BOOST_LOG_SEV(slg, D) << "[I] Timestamp updated for server " << m_sender_endpoint;
    }
	else
	{
	    // Create a new server info packet just with a timestamp
        ServerInfoPacket sip;
        sip.timeStamp = time(NULL);
        sip.key = key;

        // insert the new ServerInfoPacket into the server list
        ret = m_sipStore.addServer(ip, port, &sip);

        if(ret)
            BOOST_LOG_SEV(slg, I) << "[I] Registered new server " << m_sender_endpoint << " (total: "
                                    << m_sipStore.getNumServers() << ")";
        else
            BOOST_LOG_SEV(slg, I) << "[I] Unable to register new server " << m_sender_endpoint;
	}
}

void CMaster::handleInfoResponse(std::string ip, U32 port)
{
/*
	U8		flags;
	U32		key;
	string	gameType;
	string	missionType;
	U8		maxPlayers;
	U32		regions;
	U32		version;
	U8		infoFlags;
	U8		numBots;
	U32		cpuSpeed;
	U8		numPlayers;
	U32*    playerList;
*/

    ServerInfoPacket sip;
    bool ret;

    //ret = m_sipStore.isListed(ip, port);
    ret = m_sipStore.getServer(ip, port, &sip);

    // Is the server in the list already?
    if(!ret)
    {
        BOOST_LOG_SEV(slg, D) << "[I] Server " << m_sender_endpoint << " sending info response - not listed";
	    return;
    }

    // renew the time stamp
    sip.timeStamp      = time(NULL);

	// flags
	sip.flags          = m_packet.readU8();

    // does the server send the correct key?
    U32 key             = m_packet.readU32();

    if(key != sip.key)
    {
        BOOST_LOG_SEV(slg, D) << "[I] Info response from " << m_sender_endpoint << " has wrong key";
        return;
    }

	// gameType
    sip.gameType       = m_packet.readString();

	// missionType
    sip.missionType    = m_packet.readString();

	// maxPlayers
	sip.maxPlayers     = m_packet.readU8();

	// regions
    sip.regions        = m_packet.readU32();

	// version
	sip.version        = m_packet.readU32();

	// infoFlags
	sip.infoFlags      = m_packet.readU8();

	// numBots
	sip.numBots        = m_packet.readU8();

	// cpuSpeed
	sip.cpuSpeed       = m_packet.readU32();

	// numPlayers
	sip.numPlayers     = m_packet.readU8();

	// TODO
	// Add the GuidList

	m_sipStore.updateServer(ip, port, &sip);
}

void CMaster::sendInfoRequest(std::string ip, U32 port)
{
    ServerInfoPacket sip;
    bool ret;

    ret = m_sipStore.getServer(ip, port, &sip);

    // this should never happen!
    if(!ret)
    {
        BOOST_LOG_SEV(slg, I) << "[I] Unable to send info request to " << ip << ":" << port << " - not listed";
        return;
    }

    BOOST_LOG_SEV(slg, D) << "[S] Info request to " << m_sender_endpoint;

    // build the info request packet
    m_packet.prepare();
    m_packet.writeU8(GameMasterInfoRequest);
    m_packet.writeU8(0);
    //m_packet.writeU16(0);         // session and key are treated as one U32 by TGE 1.5.2
    //m_packet.writeU16(0);         // session and key are treated as one U32 by TGE 1.5.2
    m_packet.writeU32(sip.key);    // session and key bundled in one 4 byte packet

    // send the info request packet
    m_socket.async_send_to(
        boost::asio::buffer(m_packet, m_packet.getPacketSize()), m_sender_endpoint, boost::bind(&CMaster::packetSent,
        this, boost::asio::placeholders::error, boost::asio::placeholders::bytes_transferred)
    );
}

void CMaster::handleGameTypesRequest()
{
    std::vector<std::string>    types;
    int                         listSize;

    // GAME TYPES
    types       = m_sipStore.getGameTypes();
    listSize    = types.size();

    m_packet.prepare();

    if(listSize == 0)
    {
        m_packet.writeU8(0);
    }
    else
    {
        m_packet.writeU8(listSize);

        for(int i = 0; i < listSize; i++)
        {
            m_packet.writeU8(types[i].size());
            m_packet.writeString(types[i]);
        }
    }

    // MISSION TYPES
    types       = m_sipStore.getMissionTypes();
    listSize    = types.size();

    if(listSize == 0)
    {
        m_packet.writeU8(0);
    }
    else
    {
        m_packet.writeU8(listSize);

        for(int i = 0; i < listSize; i++)
        {
            m_packet.writeU8(types[i].size());
            m_packet.writeString(types[i]);
        }
    }

    sendGameTypesResponse();
}

void CMaster::sendGameTypesResponse()
{
    BOOST_LOG_SEV(slg, D) << "[S] Game types response to " << m_sender_endpoint;

    // send the game types response packet
    m_socket.async_send_to(
        boost::asio::buffer(m_packet, m_packet.getPacketSize()), m_sender_endpoint, boost::bind(&CMaster::packetSent,
        this, boost::asio::placeholders::error, boost::asio::placeholders::bytes_transferred)
    );
}

void CMaster::handleServerListRequest()
{
    std::multimap<std::string, U16> addresses;
    ServerFilter filter;

    U8  flags;
    U16 session;
    U16 key;
    U8  index;

    flags               = m_packet.readU8();
    session             = m_packet.readU16();
    key                 = m_packet.readU16();
    index               = m_packet.readU8();

    filter.gameType     = m_packet.readString();
    filter.missionType  = m_packet.readString();

    filter.minPlayers   = m_packet.readU8();
    filter.maxPlayers   = m_packet.readU8();
    filter.regions      = m_packet.readU32();
    filter.version      = m_packet.readU32();
    filter.infoFlags    = m_packet.readU8();
    filter.maxBots      = m_packet.readU8();
    filter.minCpuSpeed  = m_packet.readU16();

    filter.numPlayers   = m_packet.readU8();
    // TODO read player list

    // get filtered servers
    addresses = m_sipStore.getFilteredServers(filter);

    // TODO is something like this needed?
    /*
    if (filter.minPlayers < 0) filter.minPlayers = 0;
	if (filter.maxPlayers < 0) filter.maxPlayers = 0;
    if (filter.maxPlayers < filter.minPlayers) filter.maxPlayers = filter.minPlayers;
	if (filter.regions < 0) filter.regions = 0;
	if (filter.version < 0) filter.version = 0;
	if (filter.maxBots < 0) filter.maxBots = 0;
	if (filter.minCPUSpeed < 0) filter.minCPUSpeed = 0;
	if (filter.playerCount < 0) filter.playerCount = 0;
	filter.playerList = new int [filter.playerCount];
	*/

	sendServerList(session, key, addresses, index);
}

void CMaster::sendServerList(U16 session, U16 key, std::multimap<std::string, U16> addresses, U8 index)
{
    std::multimap<std::string, U16>::iterator addressesIter;
    double numServers;  // double is needed to make ceil below work

    numServers = addresses.size();

    // server packets calculation:
    // packet size is 1472 (see packet.h)
    // 1472 - 10 byte Torque header = 1462 bytes remaining for server data
    // 1462 / 6 byte for each server = max 243 servers can be sent with one packet
    // the packet number is U8 so the largest server list can be 243 servers * 254 packets = 61722
    // we're lazy and don't check if this max number is reached (no error handling here)

    U8      packetNum           = 0;
    U8      numPacketServers    = 243;
    U8      totalPackets;
    U8      packetCount         = 0;
    U8      count;

    addressesIter = addresses.begin();

    if(index != 0xFF)
    {
        BOOST_LOG_SEV(slg, D) << "[S] Resending server list packet " << index << " to " << m_sender_endpoint;
        totalPackets = 1;
        packetNum = index;
        advance(addressesIter, numPacketServers * index);
    }
    else
    {
        BOOST_LOG_SEV(slg, D) << "[S] Server list with " << numServers << " servers to " << m_sender_endpoint;
        totalPackets = ceil(numServers / numPacketServers);
    }

    for(int i = 0; i < totalPackets; i++)
    {
        m_packet.prepare();
        m_packet.writeU8(MasterServerListResponse); // packet type
        m_packet.writeU8(0);                        // flags
        m_packet.writeU16(session);                 // session
        m_packet.writeU16(key);                     // key
        m_packet.writeU8(packetNum);                // number of the current server packet
        m_packet.writeU8(totalPackets);             // total number of server packets to be sent

        packetCount++;

        if(packetCount < totalPackets)
            m_packet.writeU16(swapU16(numPacketServers));    // number of servers in the current packet
        else
        {
            if(packetNum == 0)
                m_packet.writeU16(swapU16(numServers));
            else
                m_packet.writeU16(swapU16((U16)numServers % (packetNum * numPacketServers)));
        }

        count = 1;

        while((count != numPacketServers) && (addressesIter != addresses.end()))
        {
            U32 ip = getIp((*addressesIter).first);
            U16 port = swapU16((*addressesIter).second);

            m_packet.writeU32(ip);
            m_packet.writeU16(port);

            count++;
            addressesIter++;
        }
        packetNum++;

        // send the server list packet
        m_socket.async_send_to(
            boost::asio::buffer(m_packet, m_packet.getPacketSize()), m_sender_endpoint, boost::bind(&CMaster::packetSent,
            this, boost::asio::placeholders::error, boost::asio::placeholders::bytes_transferred)
        );
    }
}

U32 CMaster::getIp(std::string ip)
{
    std::string ipStr = m_sender_endpoint.address().to_string();

    std::vector<std::string> parts;
    boost::split(parts, ipStr, boost::is_any_of("."));

    unsigned int res = (atoi(parts[0].c_str()) << 24) | (atoi(parts[1].c_str()) << 16) |
                        (atoi(parts[2].c_str()) << 8) | atoi(parts[3].c_str());

    return res;
}

U16 CMaster::swapU16(U16 port)
{
    return (((port) & 0xff) << 8) | ((U16)(port) >> 8);
}

U32 CMaster::getRandU32()
{
    // create a random number between 0x1 and 0xFFFFFFFF
    U32 ret = rand() % 0xFFFFFFFF + 0x1;

    return ret;
}

void CMaster::runCleanup()
{
    int deleted;

    BOOST_LOG_SEV(slg, D) << "[I] Cleaning up the server list";

    deleted = m_sipStore.deleteDeadServers();

    if(deleted > 0)
        BOOST_LOG_SEV(slg, I) << "[I] Deleted " << deleted << " dead server(s), " << m_sipStore.getNumServers()
                                << " remaining";

    // restart the timer
    m_cleanupTimer.expires_at(m_cleanupTimer.expires_at() + boost::posix_time::seconds(ProgramOptions.interval));
    m_cleanupTimer.async_wait(boost::bind(&CMaster::runCleanup, this));
}

void CMaster::runExec()
{
    int MAX_BUFFER = 256;
	char buffer[MAX_BUFFER];
    FILE *stream;

    // Get total server stats
    ServerTotalStats sts = m_sipStore.getServerTotalStats();

    std::stringstream command;
    command << ProgramOptions.exec << " "
            << time(NULL) << " "
            << sts.numServers << " "
            << sts.numPlayers << " "
            << sts.numBots;

    BOOST_LOG_SEV(slg, D) << "[I] Executing " << ProgramOptions.exec << " (SRV: " << sts.numServers
                            << ", PLA: " << sts.numPlayers << ", BOT: " << sts.numBots << ")";

    stream = popen(command.str().c_str(), "r");
	if (!stream)
	{
	    BOOST_LOG_SEV(slg, I) << "[E] Unable to execute external program";
		exit(1);
	}

	while (fgets(buffer, MAX_BUFFER, stream) != NULL)
	{
	    #if defined(MASTER_DEBUG)
        printf("EXEC: %s", buffer);
        #endif // DEBUG
	}

	if (ferror(stream))
        BOOST_LOG_SEV(slg, I) << "[E] There was an execution problem when launching the external program";

	pclose(stream);

    // restart the exec timer
    m_execTimer.expires_at(m_execTimer.expires_at() + boost::posix_time::minutes(ProgramOptions.execinterval));
    m_execTimer.async_wait(boost::bind(&CMaster::runExec, this));
}

#if defined(MASTER_DEBUG)
void CMaster::addFakeServers(S32 num)
{
    printf("Adding %i fake servers\n", num);

    for(int i = 0; i < num; i++)
    {
        ServerInfoPacket sip;
        sip.timeStamp = time(NULL);

        sip.flags       = 0;
        sip.key         = getRandU32();
        sip.gameType    = "FPS Starter Kit";
        sip.missionType = "Deathmatch";
        sip.maxPlayers  = 64;
        sip.regions     = 0x02000000;
        sip.version     = 0x1C070000;
        sip.infoFlags   = 0;
        sip.numBots     = 0;
        sip.cpuSpeed    = 0x6400F;
        sip.numPlayers  = 5;

        std::string ip  = "10.11.12.13";

        boost::asio::ip::address ip_addr = boost::asio::ip::address::from_string(ip);
        udp::endpoint ep(ip_addr, 29000 + i);

        m_sipStore.addServer(ip, 29000 + i, &sip);
    }
}
#endif // DEBUG
