//////////////////////////////////////////////////////////
//
// Filename: sip_store.h
// Author:   Stefan (Shaderman) Greven
// Date:     26.02.2010
//
// Description: This is the header of the
//              Server Info Packet Store class
//
//////////////////////////////////////////////////////////

#ifndef SIP_STORE_H_INCLUDED
#define SIP_STORE_H_INCLUDED

#include "main.h"

#include "sqlite3.h"

#include <boost/asio.hpp>

using boost::asio::ip::udp;

struct ServerTotalStats
{
    ServerTotalStats()
    {
        numServers  = 0;
        numPlayers  = 0;
        numBots     = 0;
    }

    int numServers;
    int numPlayers;
    int numBots;
};

struct ServerFilter
{
    //    packetIndex       Request specific page # (rest empty)
    std::string gameType;       // Game type string
    std::string missionType;    // Mission type string
    U8          minPlayers;     // At least this many players
    U8          maxPlayers;     // No more than this many
    U32         regions;        // Region mask, 0 = all
    U32         version;        // Server version, 0 = any
    U8          infoFlags;      // is this correct? Server flags (dedicated, etc), 0 = any
    U8          maxBots;        // No more than maxBots
    U16         minCpuSpeed;    // size correct or U32? At least this fast
    U8          numPlayers;     // Buddy list search
    // TODO guidlist
};

struct ServerInfoPacket
{
    // TODO are flags and infoFlags the same?
    ServerInfoPacket()
    {
        ip          = "";
        port        = 0;
        flags       = 0;
        key         = 0;
        gameType    = "";
        missionType = "";
        maxPlayers  = 0;
        regions     = 0;
        version     = 0;
        infoFlags   = 0;
        numBots     = 0;
        cpuSpeed    = 0;
        numPlayers  = 0;
    }

	// general information
	long    timeStamp;

	// server info
	std::string ip;     // this IP will be sent on server list request
	U16         port;   // this port will be sent on server list request;
	U8          flags;  // dedicated (bit 0) / password (bit 1) / Linux (bit 2) / version (bit 7)
	U32         key;
	std::string	gameType;
	std::string missionType;
	U8	        maxPlayers;
	U32	        regions;
	U32	        version;
	U8	        infoFlags;
	U8	        numBots;
	U32	        cpuSpeed;
	U8	        numPlayers;
	// TODO guidlist
};

class CSipStore
{
public:
    CSipStore();
    ~CSipStore();

    // Initialize the sip store database
    void init(std::string fileName);

    // Shutdown (close) the sip store database
    void shutdown();

    // Adds a new server to the server list. Returns the size of the server list.
    bool addServer(std::string ip, U32 port, ServerInfoPacket* sip);

    // Delete dead servers
    int deleteDeadServers();

    // get an array of filtered (matching) servers
    std::multimap<std::string, U16> getFilteredServers(ServerFilter filter);

    // Get the number of servers in the server list
    int getNumServers();

    // Get server from server list
    bool getServer(std::string ip, U32 port, ServerInfoPacket* sip);

    // Get a list of all game types
    std::vector<std::string> getGameTypes();

    // Get a list of all mission types
    std::vector<std::string> getMissionTypes();

    // Check if a server in the server list already
    bool isListed(std::string ip, U32 port);

    // Get some stats about all servers
    ServerTotalStats getServerTotalStats();

    // Sets the time stamp for a server
    bool setTimeStampAndKey(std::string ip, U32 port, U32 key);

    // Updates a servers information
    bool updateServer(std::string ip, U32 port, ServerInfoPacket* sip);

private:
    sqlite3 *m_db;
};

#endif // SIP_STORE_H_INCLUDED
