//////////////////////////////////////////////////////////
//
// Filename: sip_store.cpp
// Author:   Stefan (Shaderman) Greven
// Date:     09.03.2010
//
// Description: This is the Server Info Packet Store class
//
//////////////////////////////////////////////////////////

#include "sip_store.h"
#include "logging.h"
#include "options.h"

#include <map>

extern struct prog_opt ProgramOptions;

CSipStore::CSipStore()
{
    init("tms.db");     // TODO make filename an option
}

CSipStore::~CSipStore()
{
    sqlite3_close(m_db);
}

void CSipStore::init(std::string fileName)
{
    int         ret;
    std::string sql;
    char*       err;

    ret = sqlite3_open(fileName.c_str(), &m_db);

    if(ret)
    {
        BOOST_LOG_SEV(slg, I) << "[E] Unable to open the SQLite database file";
        sqlite3_close(m_db);
        exit(1);
    }

    if(ProgramOptions.purge == 1)
    {
        BOOST_LOG_SEV(slg, I) << "[I] Purging database";

        sql = "DROP TABLE servers;";

        ret = sqlite3_exec(m_db, sql.c_str(), NULL, NULL, &err);

        if(ret != SQLITE_OK)
        {
            BOOST_LOG_SEV(slg, I) << "[E] Unable to purge SQLite database";

            if(err != NULL)
                sqlite3_free(err);
        }
    }

    sql = "CREATE TABLE IF NOT EXISTS servers \
            ( \
                server_id INTEGER NOT NULL PRIMARY KEY AUTOINCREMENT, \
                timestamp INTEGER, \
                ip TEXT NOT NULL, \
                port INTEGER, \
                flags INTEGER, \
                key INTEGER, \
                gameType TEXT, \
                missionType TEXT, \
                maxPlayers INTEGER, \
                regions INTEGER, \
                version INTEGER, \
                infoFlags INTEGER, \
                numBots INTEGER, \
                cpuSpeed INTEGER, \
                numPlayers INTEGER \
            );";

    ret = sqlite3_exec(m_db, sql.c_str(), NULL, NULL, &err);

    if(ret != SQLITE_OK)
    {
        BOOST_LOG_SEV(slg, I) << "[E] Unable to create the SQLite database";

        if(err != NULL)
            sqlite3_free(err);
    }
}

void CSipStore::shutdown()
{
    sqlite3_close(m_db);
}

bool CSipStore::addServer(std::string ip, U32 port, ServerInfoPacket* sip)
{
    bool    ret;
    int     res;
    char*   sql;
    char*   err;

    ret = false;

    sql = sqlite3_mprintf("INSERT INTO servers VALUES(NULL, \
                                %i, \
                                '%q', \
                                %i, \
                                %i, \
                                %i, \
                                '%q', \
                                '%q', \
                                %i, \
                                %i, \
                                %i, \
                                %i, \
                                %i, \
                                %i, \
                                %i);",
                                time(NULL),
                                ip.c_str(),
                                port,
                                sip->flags,
                                sip->key,
                                sip->gameType.c_str(),
                                sip->missionType.c_str(),
                                sip->maxPlayers,
                                sip->regions,
                                sip->version,
                                sip->infoFlags,
                                sip->numBots,
                                sip->cpuSpeed,
                                sip->numPlayers
                            );

    res = sqlite3_exec(m_db, sql, NULL, NULL, &err);

    if(res != SQLITE_OK)
    {
        BOOST_LOG_SEV(slg, I) << "[E] Unable to add a new server";

        if(err != NULL)
            sqlite3_free(err);
    }
    else
    {
        ret = true;
    }

    return ret;
}

int CSipStore::deleteDeadServers()
{
    int     res;
    char*   sql;
    char*   err;
    int     changed;

    sql = sqlite3_mprintf("DELETE FROM servers WHERE timestamp < %i;", time(NULL) - ProgramOptions.lifetime);

    res = sqlite3_exec(m_db, sql, NULL, NULL, &err);

    changed = sqlite3_changes(m_db);

    if(res != SQLITE_OK)
    {
        BOOST_LOG_SEV(slg, I) << "[E] Unable to delete servers";

        if(err != NULL)
            sqlite3_free(err);
    }

    return changed;
}

std::multimap<std::string, U16> CSipStore::getFilteredServers(ServerFilter filter)
{
    char*                           sql;
    int                             res;
    int                             cols;
    sqlite3_stmt*                   stmt;
    std::multimap<std::string, U16> addressMap;
    std::stringstream               sqlString;
    char                            buffer[10];

    // GAMETYPE
    sqlString << "SELECT ip, port FROM servers WHERE gameType = ";
    sqlString << "'" << filter.gameType.c_str() << "'";

    // MISSIONTYPE
    if(filter.missionType != "Any")
        sqlString << " AND missionType = '" << filter.missionType.c_str() << "'";

    // NUMPLAYERS
    sqlString << " AND numPlayers >= ";
    sprintf(buffer, "%u", filter.minPlayers);
    sqlString << buffer;

    // MAXPLAYERS
    sqlString << " AND maxPlayers <= ";
    sprintf(buffer, "%u", filter.maxPlayers);
    sqlString << buffer;

    // REGIONS
    if(filter.regions != 0)
    {
        sqlString << " AND regions = ";
        sprintf(buffer, "%u", filter.regions);
        sqlString << buffer;
    }

    // VERSION
    if(filter.version != 0)
    {
        sqlString << " AND version = ";
        sprintf(buffer, "%u", filter.version);
        sqlString << buffer;
    }

    // INFOFLAGS
    if(filter.infoFlags != 0)
    {
        sqlString << " AND infoFlags = ";
        sprintf(buffer, "%u", filter.infoFlags);
        sqlString << buffer;
    }

    // NUMBOTS
    sqlString << " AND numBots <= ";
    sprintf(buffer, "%u", filter.maxBots);
    sqlString << buffer;

    // CPUSPEED
    sqlString << " AND cpuSpeed >= ";
    sprintf(buffer, "%u", filter.minCpuSpeed);
    sqlString << buffer;

    sql = sqlite3_mprintf(sqlString.str().c_str());

    res = sqlite3_prepare_v2(m_db, sql, -1, &stmt, 0);

    if(res != SQLITE_OK)
        BOOST_LOG_SEV(slg, I) << "[E] Unable to get filtered servers from SQLite database";
    else
    {
        res = sqlite3_step(stmt);
        cols = sqlite3_column_count(stmt);

        while(res == SQLITE_ROW)
        {
            addressMap.insert(std::pair<std::string, U32>(
                        (const char*)sqlite3_column_text(stmt, 0),
                        sqlite3_column_int(stmt, 1)));

            res = sqlite3_step(stmt);
        }
    }

    sqlite3_finalize(stmt);

    return addressMap;
}

int CSipStore::getNumServers()
{
    int             res;
    char*           sql;
    int             num;
    sqlite3_stmt*   stmt;

    num = 0;

    sql = sqlite3_mprintf("SELECT COUNT(server_id) FROM servers;");

    res = sqlite3_prepare_v2(m_db, sql, -1, &stmt, 0);

    if(res != SQLITE_OK)
        BOOST_LOG_SEV(slg, I) << "[E] Unable to get the number of servers from the SQLite database";
    else
    {
        res = sqlite3_step(stmt);

        if(res == SQLITE_ROW)
        {
            num = sqlite3_column_int(stmt, 0);

            res = sqlite3_step(stmt);
        }
    }

    return num;
}

bool CSipStore::getServer(std::string ip, U32 port, ServerInfoPacket* sip)
{
    int                 res;
    char*               sql;
    char*               err;
    sqlite3_stmt*       stmt;
    bool                ret;
    const char*         text;

    ret = false;

    sql = sqlite3_mprintf("SELECT * FROM servers WHERE ip=%Q AND port=%i;", ip.c_str(), port);

    if (SQLITE_OK == sqlite3_prepare_v2(m_db, sql, -1, &stmt, 0))
    {
        if (sqlite3_step(stmt) == SQLITE_ROW)
        {
            ret = true;

            sip->ip         = ip;
            sip->port       = port;
            sip->flags      = (U8)sqlite3_column_int(stmt, 4);
            sip->key        = (U32)sqlite3_column_int(stmt, 5);

            text = (const char*)sqlite3_column_text(stmt, 6);
            if(text != NULL)
                sip->gameType    = text;

            text = (const char*)sqlite3_column_text(stmt, 7);
            if(text != NULL)
                sip->missionType = text;

            sip->maxPlayers = (U8)sqlite3_column_int(stmt, 8);
            sip->regions    = (U32)sqlite3_column_int(stmt, 9);
            sip->version    = (U32)sqlite3_column_int(stmt, 10);
            sip->infoFlags  = (U8)sqlite3_column_int(stmt, 11);
            sip->numBots    = (U8)sqlite3_column_int(stmt, 12);
            sip->cpuSpeed   = (U32)sqlite3_column_int(stmt, 13);
            sip->numPlayers = (U8)sqlite3_column_int(stmt, 14);
        }
    }

    sqlite3_finalize(stmt);

    res = sqlite3_exec(m_db, sql, NULL, NULL, &err);

    if(res != SQLITE_OK)
    {
        BOOST_LOG_SEV(slg, I) << "[E] Unable to get a server from the SQLite database";

        if(err != NULL)
            sqlite3_free(err);
    }
    else
    {
        ret = true;
    }

    return ret;
}

std::vector<std::string> CSipStore::getGameTypes()
{
    std::vector<std::string>    gameTypes;
    char*                       sql;
    int                         res;
    sqlite3_stmt*               stmt;

    sql = sqlite3_mprintf("SELECT DISTINCT gameType FROM servers;");

    res = sqlite3_prepare_v2(m_db, sql, -1, &stmt, 0);

    if(res != SQLITE_OK)
        BOOST_LOG_SEV(slg, I) << "[E] Unable to get game types from SQLite database";
    else
    {
        res = sqlite3_step(stmt);

        while(res == SQLITE_ROW)
        {
            gameTypes.push_back((const char*)sqlite3_column_text(stmt, 0));

            res = sqlite3_step(stmt);
        }
    }

    sqlite3_finalize(stmt);

    return gameTypes;
}

std::vector<std::string> CSipStore::getMissionTypes()
{
    std::vector<std::string>    missionTypes;
    char*                       sql;
    int                         res;
    sqlite3_stmt*               stmt;

    sql = sqlite3_mprintf("SELECT DISTINCT missionType FROM servers;");

    res = sqlite3_prepare_v2(m_db, sql, -1, &stmt, 0);

    if(res != SQLITE_OK)
        BOOST_LOG_SEV(slg, I) << "[E] Unable to get game types from SQLite database";
    else
    {
        res = sqlite3_step(stmt);

        while(res == SQLITE_ROW)
        {
            missionTypes.push_back((const char*)sqlite3_column_text(stmt, 0));

            res = sqlite3_step(stmt);
        }
    }

    sqlite3_finalize(stmt);

    return missionTypes;
}

bool CSipStore::isListed(std::string ip, U32 port)
{
    char*           sql;
    sqlite3_stmt*   stmt;
    bool            ret;

    ret = false;

    sql = sqlite3_mprintf("SELECT * FROM servers WHERE ip=%Q AND port=%i;", ip.c_str(), port);

    if (SQLITE_OK == sqlite3_prepare_v2(m_db, sql, -1, &stmt, 0))
    {
        if (sqlite3_step(stmt) == SQLITE_ROW)
            ret = true;
    }

    sqlite3_finalize(stmt);

    return ret;
}

ServerTotalStats CSipStore::getServerTotalStats()
{
    ServerTotalStats    stats;
    char*               sql;
    int                 res;
    char*               err;
    char**              result;
    int                 rows;
    int                 cols;

    sql = sqlite3_mprintf("SELECT COUNT(server_id), SUM(numPlayers), SUM(numBots) FROM SERVERS;");

    res = sqlite3_get_table(m_db, sql, &result, &rows, &cols, &err);

    if(res != SQLITE_OK)
    {
        BOOST_LOG_SEV(slg, I) << "[E] Unable to get server stats from the SQLite database";

        if(err != NULL)
            sqlite3_free(err);
    }
    else
    {
        if(rows == 1 && cols == 3)
        {
            if(result[3] != NULL)
                stats.numServers    = atoi(result[3]);
            if(result[4] != NULL)
                stats.numPlayers    = atoi(result[4]);
            if(result[5] != NULL)
                stats.numBots       = atoi(result[5]);

            sqlite3_free_table(result);
        }
    }

    return stats;
}

bool CSipStore::setTimeStampAndKey(std::string ip, U32 port, U32 key)
{
    int             res;
    char*           sql;
    char*           err;
    sqlite3_stmt*   stmt;
    bool            ret;

    ret = false;

    sql = sqlite3_mprintf("UPDATE servers SET key = %i, timestamp = %i WHERE ip=%Q AND port=%i;",
                          key, time(NULL), ip.c_str(), port);

    if (SQLITE_OK == sqlite3_prepare_v2(m_db, sql, -1, &stmt, 0))
    {
        res = sqlite3_exec(m_db, sql, NULL, NULL, &err);

        if(res != SQLITE_OK)
        {
            BOOST_LOG_SEV(slg, I) << "[E] Unable to set server timestamp in the SQLite database";

            if(err != NULL)
                sqlite3_free(err);
        }
        else
        {
            ret = true;
        }
    }

    sqlite3_finalize(stmt);

    return ret;
}

bool CSipStore::updateServer(std::string ip, U32 port, ServerInfoPacket* sip)
{
    int             res;
    char*           sql;
    char*           err;
    sqlite3_stmt*   stmt;
    bool            ret;

    ret = false;

    sql = sqlite3_mprintf("UPDATE servers SET \
                          timestamp = %i, \
                          flags = %i, \
                          gameType = %Q, \
                          missionType = %Q, \
                          maxPlayers = %i, \
                          regions = %i, \
                          version = %i, \
                          infoFlags = %i, \
                          numBots = %i, \
                          cpuSpeed = %i, \
                          numPlayers = %i \
                          WHERE ip=%Q AND port=%i;",
                          time(NULL),
                          sip->flags,
                          sip->gameType.c_str(),
                          sip->missionType.c_str(),
                          sip->maxPlayers,
                          sip->regions,
                          sip->version,
                          sip->infoFlags,
                          sip->numBots,
                          sip->cpuSpeed,
                          sip->numPlayers,
                          ip.c_str(),
                          port);

    if (SQLITE_OK == sqlite3_prepare_v2(m_db, sql, -1, &stmt, 0))
    {
        res = sqlite3_exec(m_db, sql, NULL, NULL, &err);

        if(res != SQLITE_OK)
        {
            BOOST_LOG_SEV(slg, I) << "[E] Unable to update SQLite information for a server";

            if(err != NULL)
                sqlite3_free(err);
        }
        else
        {
            ret = true;
        }
    }

    sqlite3_finalize(stmt);

    return ret;
}
