//////////////////////////////////////////////////////////
//
// Filename: options.cpp
// Author:   Stefan (Shaderman) Greven
// Date:     03.01.2010
//
// Description: Program options
//
//////////////////////////////////////////////////////////

#include "options.h"
#include "logging.h"
#include "version.h"

#include <fstream>
#include <iostream>

#include <boost/program_options.hpp>
#include <boost/program_options/positional_options.hpp>

namespace po = boost::program_options;

extern struct prog_opt ProgramOptions;

bool COptions::checkArgs(int ac, char* av[])
{
    try
    {
        // command line only options
        po::options_description generic("Generic");
        generic.add_options()
            ("version,v", "         Print version string")
            ("help,h", "         Show this help message")
        ;

        // command line and/or config file options
        po::options_description networking("Networking");
        networking.add_options()
            ("address,a", po::value<std::string>(), "         IP address to listen on")
            ("port,p", po::value<int>(&ProgramOptions.port), "         Port to listen on")
        ;

        po::options_description execution("Execution");
        execution.add_options()
            ("configfile,c", po::value<std::string>(&ProgramOptions.configFile), "       Config file to use")
            ("daemonuser,d", po::value<std::string>(&ProgramOptions.daemonUser), "       Run the master server as daemon <user>")
            ("purge,u", po::value<int>(&ProgramOptions.purge)->default_value(0), "       Purge database on start (0 = no, 1 = yes)")
        ;

        po::options_description execTimer("Execution timer");
        execTimer.add_options()
            ("exec,e", po::value<std::string>(), "Script or program to execute")
            ("execinterval,t", po::value<int>(&ProgramOptions.execinterval)->default_value(TMS_EXECINTERVAL), "Execution interval (minutes)")
        ;

        po::options_description logging("Logging");
        logging.add_options()
            ("logfile,f", po::value<std::string>(&ProgramOptions.logFile), "    Name of a log file")
            ("loglevel,o", po::value<int>(&ProgramOptions.logLevel)->default_value(ProgramOptions.logLevel), "    Loglevel (1 = normal, 2 = verbose)")
            ("syslog,s", po::value<std::string>(&ProgramOptions.syslogHost), "    A syslog host (UDP port 514, local or remote)")
        ;

        po::options_description serverlist("Server list");
        serverlist.add_options()
            ("interval,i", po::value<int>(&ProgramOptions.interval)->default_value(TMS_CLEANUPINTERVAL), "  Server list cleanup interval (seconds)")
            ("lifetime,l", po::value<int>(&ProgramOptions.lifetime)->default_value(TMS_SERVERLIFETIME), "  Server lifetime (seconds)")
        ;

        po::options_description cmdline_options;
        cmdline_options.add(generic).add(networking).add(execution).add(logging).add(execTimer).add(serverlist);

        po::options_description config_file_options;
        config_file_options.add(networking).add(execution).add(logging).add(execTimer).add(serverlist);

        po::variables_map vm;
        store(po::command_line_parser(ac, av).options(cmdline_options).run(), vm);

        notify(vm);

        if (vm.count("help"))
        {
            std::cout << generic << "\n";
            std::cout << networking << "\n";
            std::cout << execution << "\n";
            std::cout << logging << "\n";
            std::cout << execTimer << "\n";
            std::cout << serverlist << "\n";
            exit(0);
        }

        if (vm.count("version"))
        {
            std::cout << "Torque Master Server " << AutoVersion::MAJOR << "."
                                                 << AutoVersion::MINOR << "."
                                                 << AutoVersion::BUILD << "\n";
            exit(0);
        }

        // config file option was supplied
        if(vm.count("configfile"))
        {
            std::ifstream ifs(ProgramOptions.configFile.c_str());
            if(ifs)
            {
                store(parse_config_file(ifs, config_file_options), vm);
                notify(vm);

                ProgramOptions.configFile = vm["configfile"].as<std::string>();

                if(ifs.is_open())
                    ifs.close();
            }
            else
            {
                std::cout << "Unable to open config file " << ProgramOptions.configFile << "!\n";
                return false;
            }
        }

        if (vm.count("address"))
            ProgramOptions.address = vm["address"].as<std::string>();

        if (ProgramOptions.address == "")
        {
            std::cout    << "At least an IP address is needed to run the master server. Use -a <address>\n"
                    << "to set an IP address. The default port 28002 can be overridden with -p <port>.\n"
                    << "You might want to use a config file with -c <config file> instead of the\n"
                    << "command line parameters.\n";

            return false;
        }

        if (vm.count("daemon"))
            ProgramOptions.daemonUser   = vm["daemon"].as<std::string>();

        if (vm.count("syslog"))
            ProgramOptions.syslogHost   = vm["syslog"].as<std::string>();

        if (vm.count("logfile"))
            ProgramOptions.logFile      = vm["logfile"].as<std::string>();

        if (vm.count("port"))
            ProgramOptions.port         = vm["port"].as<int>();

        if (vm.count("exec"))
        {
            ProgramOptions.exec         = vm["exec"].as<std::string>();

            if (vm.count("execinterval"))
                ProgramOptions.execinterval = vm["execinterval"].as<int>();
        }
    }
    catch(std::exception& e)
    {
        std::cout << e.what() << "\n";
        return false;
    }
    catch(...)
    {
        BOOST_LOG_SEV(slg, I) << "[E] Received a program option exception";
        return false;
    }

    return true;
};
