//////////////////////////////////////////////////////////
//
// Filename: main.cpp
// Author:   Stefan (Shaderman) Greven
// Date:     29.12.2009
//
// Description: This is the entry point to the
//              Master Server
//
//////////////////////////////////////////////////////////

#include "main.h"
#include "tms.h"
#include "logging.h"
#include "options.h"
#include "version.h"
#include "sip_store.h"

#include <iostream>
#include <pwd.h>

using boost::asio::ip::udp;

struct prog_opt ProgramOptions;

void makeDaemon()
{
    pid_t   parent, pid, sid;
    int     descriptor = -1;
    char    pidStr[10];
    int     ret;
    FILE*   file;

    // already a daemon?
    if (getppid() == 1)
    {
        perror("[ERROR] There's already a daemon process running");
        return;
    }

    // set the user id
    struct passwd *pw = getpwnam(ProgramOptions.daemonUser.c_str());
    if (pw)
    {
        setuid( pw->pw_uid );
    }
    else
    {
        perror("[ERROR] Can't run with that daemon user");
    }

    pid = fork();

    // fork problem?
    if (pid < 0)
    {
        perror("[ERROR] Can't fork daemon process");
        exit(1);
    }

    // the parent exits
    if (pid > 0)
    {
        alarm(2);
        pause();
    }

    // Running as child process from here...
    parent = getppid();

    // The daemon can be shut down properly with SIGINT
    signal(SIGTERM,SIG_DFL); // Default signal handler
    signal(SIGCHLD,SIG_DFL); // A child dies
    signal(SIGTSTP,SIG_IGN); // Ignore keyboard input
    signal(SIGTTOU,SIG_IGN); // Ignore tty output stuff
    signal(SIGTTIN,SIG_IGN); // Ignore tty input stuff
    signal(SIGHUP, SIG_IGN); // Ignore hangup signal

    // set file permissions
    umask(0);

    // get a new process group
    sid = setsid();
    if (sid < 0)
    {
        BOOST_LOG_SEV(slg, I) << "[E] Can't create new daemon session";
        exit(1);
    }

    // change to the users home dir
    if (chdir(pw->pw_dir) < 0)
    {
        BOOST_LOG_SEV(slg, I) << "[E] Can't change to daemon user directory";
        exit(1);
    }

    // create the pid file containing the pid of the daemon
    descriptor = open(TMS_PID_FILE, O_RDWR|O_CREAT, 0640);
    if (descriptor < 0)
    {
        BOOST_LOG_SEV(slg, I) << "[E] Can't access pid file";
        exit(1);
    }
    else
    {
        sprintf(pidStr, "%d\n", getpid());
        ret = write(descriptor, pidStr, strlen(pidStr));
        if(ret <= 0)
        {
            BOOST_LOG_SEV(slg, I) << "[E] Can't write to pid file";
            exit(1);
        }
    }

    // redirect standard files to /dev/null
    file = freopen( "/dev/null", "r", stdin);
    if(file == NULL)
    {
        BOOST_LOG_SEV(slg, I) << "[E] Can't redirect stdin";
        exit(1);
    }

    file = freopen( "/dev/null", "w", stdout);
    if(file == NULL)
    {
        BOOST_LOG_SEV(slg, I) << "[E] Can't redirect stdout";
        exit(1);
    }

    file = freopen( "/dev/null", "w", stderr);
    if(file == NULL)
    {
        BOOST_LOG_SEV(slg, I) << "[E] Can't redirect stderr";
        exit(1);
    }

    kill (parent, SIGTERM);
}

int main(int ac, char* av[])
{
    sigset_t    set;
    int         sig;
    int         *sigPtr = &sig;
    int         retVal;

    sigemptyset(&set);
    sigaddset(&set, SIGINT);
    sigprocmask(SIG_BLOCK, &set, NULL);

    // check program command line options
    COptions options;
    bool ret = options.checkArgs(ac, av);

    // there was something wrong with the program options
    if(!ret)
    {
        return 1;
    }

    init_logging();

    // run as daemon?
    if (ProgramOptions.daemonUser != "")
        makeDaemon();

    // initialize random seed
    srand( time(NULL) );

    BOOST_LOG_SEV(slg, I) << "[I] Starting Torque Master Server " << AutoVersion::MAJOR
                                                                  << "." << AutoVersion::MINOR
                                                                  << "." << AutoVersion::BUILD;

    // we need a sip store
    CSipStore sip_store;

    boost::asio::ip::address ip_addr = boost::asio::ip::address::from_string(ProgramOptions.address);

    // start the networking part
    try
    {
        // socket io_service
        boost::asio::io_service ios_socket;

        // cleanup timer io_service
        boost::asio::io_service ios_cleanupTimer;

        // exec timer io_service
        boost::asio::io_service ios_execTimer;

        // create a master server instance
        CMaster master(sip_store, ios_socket, ios_cleanupTimer, ios_execTimer, ip_addr, ProgramOptions.port);

        // start the cleanup timer in its own thread
        BOOST_LOG_SEV(slg, D) << "[I] Starting server list cleanup thread";
        boost::thread threadTimer(boost::bind(&boost::asio::io_service::run, &ios_cleanupTimer));

        // start the exec timer in its own thread
        boost::thread threadExec;
        if(ProgramOptions.exec != "")
        {
            BOOST_LOG_SEV(slg, D) << "[I] Starting execution timer thread";
            boost::thread threadExec(boost::bind(&boost::asio::io_service::run, &ios_execTimer));
        }

        // start the socket io_service in its own thread
        BOOST_LOG_SEV(slg, D) << "[I] Starting socket I/O thread";
        boost::thread threadSocket(boost::bind(&boost::asio::io_service::run, &ios_socket));

        retVal = sigwait(&set,sigPtr);

        if(retVal == -1)
            BOOST_LOG_SEV(slg, I) << "[E] SIGWAIT call failed";
        else
        {
            if (*sigPtr == 2)
            {
                // delete the pid file
                if(ProgramOptions.daemonUser != "")
                    if (remove(TMS_PID_FILE) != 0)
                        BOOST_LOG_SEV(slg, I) << "[E] Unable to delete the pid file";

                // shut down the threads
                BOOST_LOG_SEV(slg, I) << "[I] Shutting down Torque Master Server";

                BOOST_LOG_SEV(slg, D) << "[I] Stopping cleanup timer";
                ios_cleanupTimer.stop();
                threadTimer.join();

                BOOST_LOG_SEV(slg, D) << "[I] Stopping execution timer";
                ios_execTimer.stop();
                if(ProgramOptions.exec != "")
                    threadExec.join();

                BOOST_LOG_SEV(slg, D) << "[I] Stopping socket IO";
                ios_socket.stop();
                threadSocket.join();

                // TODO
                BOOST_LOG_SEV(slg, D) << "[I] Closing server info database";
                sip_store.shutdown();
            }
        }

    }
    catch (std::exception& e)
    {
        BOOST_LOG_SEV(slg, I) << e.what();
    }

    return 0;
}
