//////////////////////////////////////////////////////////
//
// Filename: logging.cpp
// Author:   Stefan (Shaderman) Greven
// Date:     29.12.2009
//
// Description: Logging class
//
//////////////////////////////////////////////////////////

#include "logging.h"
#include "options.h"

#include <iostream>
#include <fstream>

#include <boost/shared_ptr.hpp>
#include <boost/log/attributes.hpp>
#include <boost/log/formatters.hpp>
#include <boost/log/sinks.hpp>
#include <boost/log/utility/init/to_console.hpp>

namespace attrs     = boost::log::attributes;
namespace flt       = boost::log::filters;
namespace fmt       = boost::log::formatters;
namespace keywords  = boost::log::keywords;
namespace logging   = boost::log;
namespace sinks     = boost::log::sinks;

using boost::shared_ptr;

extern struct prog_opt ProgramOptions;

void init_logging()
{
    // get the logging core
    boost::shared_ptr< logging::core > core = logging::core::get();

    if(ProgramOptions.daemonUser == "")
    {
        //
        // CONSOLE LOGGING
        //

        // Initialize logging to std::clog
        boost::shared_ptr< sinks::synchronous_sink< sinks::text_ostream_backend > >
            consoleSink = logging::init_log_to_console();

        shared_ptr< logging::attribute > pTimeStamp(new attrs::local_clock());
        logging::core::get()->add_global_attribute("TimeStamp", pTimeStamp);

        consoleSink->locked_backend()->set_formatter
        (
            fmt::format("%1% %2%")
            % fmt::date_time< boost::posix_time::ptime >("TimeStamp", "%m-%d-%Y %H:%M:%S")
            % fmt::message()
        );

        consoleSink->locked_backend()->auto_flush(true);

        if (ProgramOptions.logLevel == 2)
        {
            consoleSink->set_filter
            (
                flt::attr< severity_level >("Severity", std::nothrow) >= D
            );
        }
        else
        {
            consoleSink->set_filter
            (
                flt::attr< severity_level >("Severity", std::nothrow) >= I
            );
        }

        //
        // FILE LOGGING
        //

        if(ProgramOptions.logFile != "")
        {
            typedef sinks::synchronous_sink< sinks::text_file_backend > text_sink;
            boost::shared_ptr< text_sink > fileSink = boost::make_shared< text_sink >
            (
                keywords::file_name = ProgramOptions.logFile,
                keywords::open_mode = (std::ios::out | std::ios::app),
                keywords::auto_flush = true
            );

            fileSink->locked_backend()->set_formatter
            (
                fmt::format("%1% %2%")
                % fmt::date_time< boost::posix_time::ptime >("TimeStamp", "%m-%d-%Y %H:%M:%S")
                % fmt::message()
            );

            fileSink->locked_backend()->auto_flush(true);

            if (ProgramOptions.logLevel == 2)
            {
                fileSink->set_filter
                (
                    flt::attr< severity_level >("Severity", std::nothrow) >= D
                );
            }
            else
            {
                fileSink->set_filter
                (
                    flt::attr< severity_level >("Severity", std::nothrow) >= I
                );
            }

            // Register the sink in the logging core
            core->add_sink(fileSink);
        }
    }

    //
    // SYSLOG LOGGING
    //

    if(ProgramOptions.syslogHost != "")
    {
        // Create a syslog sink
        shared_ptr< sinks::synchronous_sink< sinks::syslog_backend > > syslogSink
        (
            new sinks::synchronous_sink< sinks::syslog_backend >()
        );

        syslogSink->locked_backend()->set_formatter
        (
            fmt::format("tms: %1%")
            % fmt::message()
        );

        // map custom levels to syslog levels
        sinks::syslog::custom_severity_mapping< severity_level > mapping("Severity");
        mapping[I] = sinks::syslog::info;
        mapping[D] = sinks::syslog::debug;

        syslogSink->locked_backend()->set_severity_mapper(mapping);

        // syslog server address
        syslogSink->locked_backend()->set_target_address("localhost");

        if (ProgramOptions.logLevel == 2)
        {
            syslogSink->set_filter
            (
                flt::attr< severity_level >("Severity", std::nothrow) >= D
            );
        }
        else
        {
            syslogSink->set_filter
            (
                flt::attr< severity_level >("Severity", std::nothrow) >= I
            );
        }

        // Add the sink to the core
        core->add_sink(syslogSink);

        BOOST_LOG_SEV(slg, I) << "[I] Logging to syslog host " << ProgramOptions.syslogHost;
    }
}
